<?php
/*
 * Copyright (c) 2017 Teratech
 * All rights reserved.
 * Date: 7/30/2017
 * Time: 11:20 PM
 */


namespace common\helpers;

use common\models\Certificate;
use common\models\SSLModel;
use Yii;
use yii\base\Exception;

/**
 * Class SslFactory
 *
 * Description of SslFactory
 *
 * @author Ramadan Juma (ramaj93@yahoo.com)
 *
 * @package common\helpers
 */
class SSLFactory
{
    /**
     * @param string $private_key
     * @param string $cacert
     * @param SSLModel $model
     * @return array
     * @throws Exception
     */
    public static function sign($private_key, $cacert, $model)
    {
        $req_key = openssl_pkey_new();
        if (openssl_pkey_export($req_key, $out_key)) {
            $dn = array(
                "countryName" => $model->countryName,
                "stateOrProvinceName" => $model->stateOrProvinceName,
                "organizationName" => $model->organizationName,
                "organizationalUnitName" => $model->organizationalUnitName,
                "commonName" => $model->commonName,
                'localityName' => $model->localityName,
                'emailAddress' => $model->emailAddress
            );
            $args = [];
            $args['config'] = realpath(__DIR__ . "/../config/openssl.cfg");
            $args['x509_extensions'] = $model->extension;
            $req_csr = openssl_csr_new($dn, $req_key, $args);
            $req_cert = openssl_csr_sign($req_csr, $cacert, $private_key, $model->duration, $args, $model->serialNo);
            if ($req_cert != false) {
                if (openssl_x509_export($req_cert, $out_cert)) {
                    return [
                        'key' => $out_key,
                        'cert' => $out_cert
                    ];
                }
            }
        }
        $error = "";
        while (($e = openssl_error_string()) !== false) {
            $error .= "\n";
        }
        throw new Exception($error);
    }

    /**
     * Signs a certificate with the vendor CA
     * @param Certificate $certificate
     * @param SSLModel $model
     * @return array
     */
    public static function vendorSign($certificate, $model)
    {
        $key = $certificate->private_key;
        $ca = $certificate->content;
        return self::sign($key, $ca, $model);
    }

    /**
     * Signs a certificate with the root CA
     * @param $model
     * @return array
     */
    public static function rootSign($model)
    {
        $ca = "file://" . Yii::getAlias("@common/keys/root.cer");
        $key = "file://" . Yii::getAlias("@common/keys/root_key.pem");

        return self::sign($key, $ca, $model);
    }

    /**
     * If successfully returns a new signed key pair ina array
     *
     * ```php
     *
     *  return [
     *              'key'=>'----BEGIN PRIVATE..,
     *              'cert'=>'----BEGIN CERT...
     *          ]
     *
     * ```
     * @param SSLModel $model
     * @return array
     * @throws Exception
     */
    public static function rootSignEx($model)
    {
        $ca = "file://" . Yii::getAlias("@common/keys/root.cer");
        $key = "file://" . Yii::getAlias("@common/keys/root_key.pem");
        $req_key = openssl_pkey_new();
        if (openssl_pkey_export($req_key, $out_key)) {
            $dn = array(
                "countryName" => $model->countryName,
                "stateOrProvinceName" => $model->stateOrProvinceName,
                "organizationName" => $model->organizationName,
                "organizationalUnitName" => $model->organizationalUnitName,
                "commonName" => $model->commonName,
                'localityName' => $model->localityName,
                'emailAddress' => $model->emailAddress
            );
            $args = null;// [];
//            $args['config'] = realpath(__DIR__ . "/openssl.cfg");
//            $args['x509_extensions'] = $model->extension;
            $req_csr = openssl_csr_new($dn, $req_key, $args);
            $req_cert = openssl_csr_sign($req_csr, $ca, $key, 365);

            if ($req_cert != false) {
                if (openssl_x509_export($req_cert, $out_cert)) {
                    return [
                        'key' => $out_key,
                        'cert' => $out_cert
                    ];
                }
            }
        }
        $error = "";
        while (($e = openssl_error_string()) !== false) {
            $error .= "\n";
        }
        throw new Exception($error);
    }
}